<?php

namespace App\Http\Controllers;

use App\Config;
use App\Gst;
use App\Menu;
use App\Movie;
use App\Package;
use App\PpvPurchase;
use App\Season;
use App\TvSeries;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Log;
use KingFlamez\Rave\Facades\Rave as Flutterwave;

class FlutterwaveController extends Controller
{
    public function initialize(Request $request)
    {
        try {
            Log::info('Initializing Flutterwave payment', $request->all());

            $reference = Flutterwave::generateReference();

            $type = $request->type;
            $plan_id = $request->plan_id ?? null;
            $movie_id = $request->movie_id ?? null;
            $series_id = $request->series_id ?? null;

            $amount = $request->amount;
            $email = $request->email;
            $phone = $request->phone;
            $name = $request->name;

            Session::put('flutterwave_context', compact('type', 'plan_id', 'movie_id', 'series_id'));

            $data = [
                'payment_options' => 'card,banktransfer',
                'amount' => $amount,
                'email' => $email,
                'tx_ref' => $reference,
                'currency' => "NGN",
                'redirect_url' => route('flutterrave.callback'),
                'customer' => [
                    'email' => $email,
                    'phone_number' => $phone,
                    'name' => $name,
                ],
                'customizations' => [
                    'title' => __('Payment'),
                    'description' => __('PPV or Subscription Purchase'),
                ],
            ];

            $payment = Flutterwave::initializePayment($data);

            if ($payment['status'] !== 'success') {
                Log::error('Flutterwave payment initialization failed', ['response' => $payment]);
                return redirect('/')->with('error', __('Unable to initialize payment.'));
            }

            Log::info('Flutterwave payment initialized successfully', ['redirect_url' => $payment['data']['link']]);
            return redirect($payment['data']['link']);
        } catch (\Exception $e) {
            Log::error('Exception in Flutterwave initialize', ['error' => $e->getMessage()]);
            return redirect('/')->with('error', __('Something went wrong. Please try again.'));
        }
    }

    public function callback()
    {
        try {
            Log::info('Flutterwave callback hit', ['request' => request()->all()]);

            $status = request()->status;

            if ($status === 'successful') {
                $transactionID = Flutterwave::getTransactionIDFromCallback();
                $data = Flutterwave::verifyTransaction($transactionID);

                Log::info('Flutterwave transaction verified', ['data' => $data]);

                $config = Config::findOrFail(1);
                $gst = Gst::first();
                $base_currency = $config->currency_code ?? 'INR'; // Fallback to INR if null
                $auth = Auth::user();
                $payment_id = $data['data']['tx_ref'];
                $payment_amount = $data['data']['amount'];
                $payment_method = 'flutterwave';
                $payment_status = 1;

                $context = Session::get('flutterwave_context');
                if (!$context || !isset($context['type'])) {
                    Log::error('Invalid or null flutterwave_context', ['context' => $context]);
                    return redirect('/')->with('error', __('Invalid transaction context'));
                }

                $type = $context['type'];

                if ($type === 'subscription') {
                    $plan = Package::findOrFail($context['plan_id']);
                    Log::info('Processing subscription for plan', ['plan_id' => $plan->id]);

                    return (new SubscriptionController)->subscribe(
                        $payment_id,
                        $payment_method,
                        $plan->id,
                        $payment_status,
                        $payment_amount
                    );
                }

                if ($type === 'rental') {
                    $movie_id = $context['movie_id'] ?? null;
                    if (!$movie_id) {
                        Log::error('Null movie_id for rental', ['context' => $context]);
                        return redirect('/')->with('error', __('Invalid movie ID'));
                    }

                    $movie = Movie::findOrFail($movie_id);
                    if (!$movie->hours || !$movie->slug) {
                        Log::error('Null movie hours or slug', [
                            'movie_id' => $movie_id,
                            'hours' => $movie->hours,
                            'slug' => $movie->slug
                        ]);
                        return redirect('/')->with('error', __('Invalid movie data'));
                    }

                    $converted_price = Session::has('current_currency')
                        ? round(currency($movie->price, $from = $base_currency, $to = ucfirst(Session::get('current_currency')), $format = false), 2)
                        : round($movie->price, 2);

                    $converted_offer_price = null;
                    if (isset($movie->offer_price) && $movie->offer_price > 0) {
                        $converted_offer_price = Session::has('current_currency')
                            ? round(currency($movie->offer_price, $from = $base_currency, $to = ucfirst(Session::get('current_currency')), $format = false), 2)
                            : round($movie->offer_price, 2);
                    }

                    $base_price = $converted_offer_price ?? $converted_price;

                    $gst_amount = 0;
                    $total_with_gst = $base_price;
                    if ($gst && $gst->gst_enable && isset($gst->gst_per) && is_numeric($gst->gst_per)) {
                        $gst_amount = ($base_price * $gst->gst_per) / 100;
                        $total_with_gst = $base_price + $gst_amount;
                    } else {
                        Log::info('GST disabled or null', [
                            'gst_enable' => $gst->gst_enable ?? 'null',
                            'gst_per' => $gst->gst_per ?? 'null'
                        ]);
                    }

                    Log::info('Movie rental payment details', [
                        'movie_id' => $movie->id,
                        'movie_title' => $movie->title,
                        'base_price' => $base_price,
                        'gst_percentage' => $gst ? ($gst->gst_enable ? $gst->gst_per : 0) : 0,
                        'gst_amount' => $gst_amount,
                        'total_with_gst' => $total_with_gst,
                        'payment_amount' => $payment_amount,
                        'converted_price' => $converted_price,
                        'converted_offer_price' => $converted_offer_price,
                        'current_currency' => Session::get('current_currency') ?? $base_currency
                    ]);

                    if (abs($payment_amount - $total_with_gst) > 0.01) {
                        Log::error('Payment amount mismatch for movie rental', [
                            'expected' => $total_with_gst,
                            'received' => $payment_amount,
                            'difference' => $total_with_gst - $payment_amount
                        ]);

                        if (abs($payment_amount - $total_with_gst) > 1.00) {
                            return redirect()->back()->with('error', __('Payment amount mismatch. Please try again.'));
                        }
                    }

                    $purchase = PpvPurchase::create([
                        'user_id' => $auth->id,
                        'movie_id' => $movie->id,
                        'payment_id' => $payment_id,
                        'gst' => $gst ? ($gst->gst_enable ? $gst->gst_per : 0) : 0,
                        'price' => $converted_price,
                        'offer_price' => $converted_offer_price ?? $converted_price,
                        'hours' => $movie->hours,
                        'expires_at' => now()->addHours($movie->hours),
                        'type' => 'movie',
                        'payment_type' => 'Flutterwave',
                        'status' => 1,
                    ]);

                    Log::info('PPV purchase created for movie', ['purchase_id' => $purchase->id]);
                    Session::forget('flutterwave_context');

                    $route = getSubscription()->getData()->subscribed
                        ? 'movie/detail/'
                        : 'movie/guest/detail/';
                    return redirect()->to($route . $movie->slug)->with('success', __('Movie purchase successful!'));
                }

                if ($type === 'rental_series') {
                    $series_id = $context['series_id'] ?? null;
                    if (!$series_id) {
                        Log::error('Null series_id for rental_series', ['context' => $context]);
                        return redirect('/')->with('error', __('Invalid season ID'));
                    }

                    $series = Season::findOrFail($series_id);
                    if (!$series->hours || !$series->season_slug) {
                        Log::error('Null series hours or season_slug', [
                            'series_id' => $series_id,
                            'hours' => $series->hours,
                            'season_slug' => $series->season_slug
                        ]);
                        return redirect('/')->with('error', __('Invalid season data'));
                    }

                    $converted_price = Session::has('current_currency')
                        ? round(currency($series->price, $from = $base_currency, $to = ucfirst(Session::get('current_currency')), $format = false), 2)
                        : round($series->price, 2);

                    $converted_offer_price = null;
                    if (isset($series->offer_price) && $series->offer_price > 0) {
                        $converted_offer_price = Session::has('current_currency')
                            ? round(currency($series->offer_price, $from = $base_currency, $to = ucfirst(Session::get('current_currency')), $format = false), 2)
                            : round($series->offer_price, 2);
                    }

                    $base_price = $converted_offer_price ?? $converted_price;

                    $gst_amount = 0;
                    $total_with_gst = $base_price;
                    if ($gst && $gst->gst_enable && isset($gst->gst_per) && is_numeric($gst->gst_per)) {
                        $gst_amount = ($base_price * $gst->gst_per) / 100;
                        $total_with_gst = $base_price + $gst_amount;
                    } else {
                        Log::info('GST disabled or null', [
                            'gst_enable' => $gst->gst_enable ?? 'null',
                            'gst_per' => $gst->gst_per ?? 'null'
                        ]);
                    }

                    Log::info('Season rental payment details', [
                        'season_id' => $series->id,
                        'season_title' => 'Season ' . $series->season_no,
                        'base_price' => $base_price,
                        'gst_percentage' => $gst ? ($gst->gst_enable ? $gst->gst_per : 0) : 0,
                        'gst_amount' => $gst_amount,
                        'total_with_gst' => $total_with_gst,
                        'payment_amount' => $payment_amount,
                        'converted_price' => $converted_price,
                        'converted_offer_price' => $converted_offer_price,
                        'current_currency' => Session::get('current_currency') ?? $base_currency
                    ]);

                    if (abs($payment_amount - $total_with_gst) > 0.01) {
                        Log::error('Payment amount mismatch for season rental', [
                            'expected' => $total_with_gst,
                            'received' => $payment_amount,
                            'difference' => $total_with_gst - $payment_amount
                        ]);

                        if (abs($payment_amount - $total_with_gst) > 1.00) {
                            return redirect()->back()->with('error', __('Payment amount mismatch. Please try again.'));
                        }
                    }

                    $purchase = PpvPurchase::create([
                        'user_id' => $auth->id,
                        'season_id' => $series->id,
                        'payment_id' => $payment_id,
                        'gst' => $gst ? ($gst->gst_enable ? $gst->gst_per : 0) : 0,
                        'price' => $converted_price,
                        'offer_price' => $converted_offer_price ?? $converted_price,
                        'hours' => $series->hours,
                        'expires_at' => now()->addHours($series->hours),
                        'type' => 'series',
                        'payment_type' => 'Flutterwave',
                        'status' => 1,
                    ]);

                    Log::info('PPV purchase created for season', ['purchase_id' => $purchase->id]);
                    Session::forget('flutterwave_context');

                    $route = getSubscription()->getData()->subscribed
                        ? 'show/detail/'
                        : 'show/guest/detail/';
                    return redirect()->to($route . $series->season_slug)->with('success', __('Season rental successful!'));
                }

                Log::error('Invalid payment type received', ['type' => $type]);
                return redirect('/')->with('error', __('Invalid payment type.'));
            }

            Log::warning('Flutterwave callback: payment not successful or cancelled', ['status' => $status]);
            return redirect('account/purchaseplan')->with('error', __('Payment was cancelled or failed.'));
        } catch (\Exception $e) {
            Log::error('Exception in Flutterwave callback', ['error' => $e->getMessage()]);
            return redirect('account/purchaseplan')->with('error', __('Something went wrong with the payment. Please contact support.'));
        }
    }
}